package cn.com.titan.eureka.api.common;

import java.util.List;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import javax.naming.event.ObjectChangeListener;

/**
 * @Title: LeeJSONResult.java
 * @Package com.lee.utils
 * @Description: 自定义响应数据结构
 * 这个类是提供给门户，ios，安卓，微信商城用的
 * 门户接受此类数据后需要使用本类的方法转换成对于的数据类型格式（类，或者list）
 * 其他自行处理
 * 200：表示成功
 * 500：表示错误，错误信息在resultMsg字段中
 * 501：bean验证错误，不管多少个错误都以map形式返回
 * 502：拦截器拦截到用户token出错
 * 555：异常抛出信息
 * Copyright: Copyright (c) 2016
 * Company:Nathan.Lee.Salvatore
 * @author leechenxiang
 * @date 2016年4月22日 下午8:33:36
 * @version V1.0
 */
public class JSONResult {
    // 定义jackson对象
    private static final ObjectMapper MAPPER = new ObjectMapper();
    // 响应业务状态
    private String resultCode;

    // 总数
    private Integer totalCount;

    // 响应消息
    private String resultMsg;
    // 响应中的数据
    private Object data;

    public static JSONResult build(String resultCode, String resultMsg, Integer totalCount, Object data) {
        return new JSONResult(resultCode, resultMsg, totalCount, data);
    }

    public static JSONResult errorException(String resultMsg) {
        return new JSONResult("555", resultMsg, 0, null);
    }

    public JSONResult() {
    }

    public JSONResult(String resultCode, String resultMsg, Integer totalCount, Object data) {
        this.resultCode = resultCode;
        this.resultMsg = resultMsg;
        this.totalCount = totalCount;
        this.data = data;
    }

    public JSONResult(Integer totalCount, Object data) {
        this.resultCode = "200";
        this.resultMsg = "OK";
        this.totalCount = totalCount;
        this.data = data;
    }

    public JSONResult(ExceptionMsg exceptionMsg) {
        this.resultCode = exceptionMsg.getCode();
        this.resultMsg = exceptionMsg.getMsg();
        this.totalCount = 0;
        this.data = null;
    }

    //如果有额外的错误信息
    public JSONResult(ExceptionMsg exceptionMsg, String errorInfo) {
        this.resultCode = exceptionMsg.getCode();
        this.resultMsg = exceptionMsg.getMsg() + errorInfo;
        this.totalCount = 0;
        this.data = null;
    }

    public JSONResult(ExceptionMsg exceptionMsg, Object data) {
        this.resultCode = exceptionMsg.getCode();
        this.resultMsg = exceptionMsg.getMsg();
        this.totalCount = 1;
        this.data = data;
    }

    public String getresultCode() {
        return resultCode;
    }

    public void setresultCode(String resultCode) {
        this.resultCode = resultCode;
    }

    public String getresultMsg() {
        return resultMsg;
    }

    public void setresultMsg(String resultMsg) {
        this.resultMsg = resultMsg;
    }

    public Integer getTotalCount() {
        return totalCount;
    }

    public void setTotalCount(Integer totalCount) {
        this.totalCount = totalCount;
    }

    public Object getData() {
        return data;
    }

    public void setData(Object data) {
        this.data = data;
    }

    /**
     * @Description: 将json结果集转化为LeeJSONResult对象
     * 需要转换的对象是一个类
     * @param jsonData
     * @param clazz
     * @return
     * @author leechenxiang
     * @date 2016年4月22日 下午8:34:58
     */
    public static JSONResult formatToPojo(Integer totalCount, String jsonData, Class<?> clazz) {
        try {
            if (clazz == null) {
                return MAPPER.readValue(jsonData, JSONResult.class);
            }
            JsonNode jsonNode = MAPPER.readTree(jsonData);
            JsonNode data = jsonNode.get("data");
            Object obj = null;
            if (clazz != null) {
                if (data.isObject()) {
                    obj = MAPPER.readValue(data.traverse(), clazz);
                } else if (data.isTextual()) {
                    obj = MAPPER.readValue(data.asText(), clazz);
                }
            }
            return build(jsonNode.get("resultCode").toString(), jsonNode.get("resultMsg")
                    .asText(), totalCount, obj);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * @Description: 没有object对象的转化
     * @param json
     * @return
     * @author leechenxiang
     * @date 2016年4月22日 下午8:35:21
     */
    public static JSONResult format(String json) {
        try {
            return MAPPER.readValue(json, JSONResult.class);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * @Description: Object是集合转化
     * 需要转换的对象是一个list
     * @param jsonData
     * @param clazz
     * @return
     * @author leechenxiang
     * @date 2016年4月22日 下午8:35:31
     */
    public static JSONResult formatToList(Integer totalCount, String jsonData, Class<?> clazz) {
        try {
            JsonNode jsonNode = MAPPER.readTree(jsonData);
            JsonNode data = jsonNode.get("data");
            Object obj = null;
            if (data.isArray() && data.size() > 0) {
                obj = MAPPER.readValue(
                        data.traverse(),
                        MAPPER.getTypeFactory().constructCollectionType(
                                List.class, clazz));
            }
            return build(jsonNode.get("resultCode").toString(), jsonNode.get("resultMsg")
                    .asText(), totalCount, obj);
        } catch (Exception e) {
            return null;
        }
    }


}
